/* 
author: 前端川
link: www.vsqdc.cn
description: 文艺爱好者专属导航平台交互脚本
*/

// DOM加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 移动端菜单切换
    const menuBtn = document.getElementById('menuBtn');
    const mobileNav = document.getElementById('mobileNav');
    
    // 搜索功能
    const searchInput = document.getElementById('searchInput');
    const searchBtn = document.getElementById('searchBtn');
    
    // 导航链接
    const navLinks = document.querySelectorAll('.nav-link, .mobile-nav__link');
    
    // 分类卡片
    const categoryCards = document.querySelectorAll('.category-card');
    
    // 网站卡片
    const websiteCards = document.querySelectorAll('.website-card');
    
    // 初始化所有功能
    initMobileMenu();
    initSearch();
    initNavLinks();
    initCategoryCards();
    initWebsiteCards();
    initSmoothScroll();
    
    /**
     * 初始化移动端菜单
     */
    function initMobileMenu() {
        if (menuBtn && mobileNav) {
            menuBtn.addEventListener('click', function() {
                // 切换菜单按钮状态
                this.classList.toggle('active');
                // 切换移动端导航显示
                mobileNav.classList.toggle('active');
                // 禁止/允许页面滚动
                document.body.style.overflow = mobileNav.classList.contains('active') ? 'hidden' : '';
            });
            
            // 点击移动端导航链接后关闭菜单
            const mobileLinks = mobileNav.querySelectorAll('.mobile-nav__link');
            mobileLinks.forEach(link => {
                link.addEventListener('click', function() {
                    menuBtn.classList.remove('active');
                    mobileNav.classList.remove('active');
                    document.body.style.overflow = '';
                });
            });
        }
    }
    
    /**
     * 初始化搜索功能
     */
    function initSearch() {
        if (searchInput && searchBtn) {
            // 点击搜索按钮
            searchBtn.addEventListener('click', function() {
                performSearch();
            });
            
            // 按下回车键
            searchInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    performSearch();
                }
            });
        }
    }
    
    /**
     * 执行搜索操作
     */
    function performSearch() {
        const searchTerm = searchInput.value.trim();
        if (searchTerm) {
            console.log('搜索关键词:', searchTerm);
            // 这里可以添加实际的搜索逻辑
            alert(`正在搜索: ${searchTerm}`);
        } else {
            alert('请输入搜索关键词');
        }
    }
    
    /**
     * 初始化导航链接
     */
    function initNavLinks() {
        // 高亮当前页面的导航链接
        const currentUrl = window.location.href;
        navLinks.forEach(link => {
            if (link.href === currentUrl) {
                link.classList.add('active');
            }
            
            // 导航链接悬停效果
            link.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
            });
            
            link.addEventListener('mouseleave', function() {
                if (!this.classList.contains('active')) {
                    this.style.transform = 'translateY(0)';
                }
            });
        });
    }
    
    /**
     * 初始化分类卡片
     */
    function initCategoryCards() {
        categoryCards.forEach(card => {
            // 添加鼠标进入效果
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-10px) scale(1.02)';
            });
            
            // 添加鼠标离开效果
            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0) scale(1)';
            });
        });
    }
    
    /**
     * 初始化网站卡片
     */
    function initWebsiteCards() {
        websiteCards.forEach(card => {
            // 添加鼠标进入效果
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-5px)';
            });
            
            // 添加鼠标离开效果
            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });
        });
    }
    
    /**
     * 初始化平滑滚动
     */
    function initSmoothScroll() {
        // 为所有锚点链接添加平滑滚动
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                
                const targetId = this.getAttribute('href');
                if (targetId === '#') return;
                
                const targetElement = document.querySelector(targetId);
                if (targetElement) {
                    // 平滑滚动到目标位置
                    targetElement.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });
    }
    
    /**
     * 初始化页面滚动效果
     */
    function initScrollEffects() {
        // 监听页面滚动事件
        window.addEventListener('scroll', function() {
            // 头部阴影效果增强
            const header = document.querySelector('.header');
            if (header) {
                if (window.scrollY > 50) {
                    header.style.boxShadow = '0 4px 20px rgba(0, 0, 0, 0.15)';
                } else {
                    header.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
                }
            }
            
            // 元素进入视口动画
            animateOnScroll();
        });
    }
    
    /**
     * 元素进入视口动画
     */
    function animateOnScroll() {
        // 获取所有需要动画的元素
        const animatedElements = document.querySelectorAll('.category-card, .website-card, .featured-card');
        
        animatedElements.forEach(element => {
            const elementTop = element.getBoundingClientRect().top;
            const elementVisible = 150;
            
            // 当元素进入视口时添加动画类
            if (elementTop < window.innerHeight - elementVisible) {
                element.style.opacity = '1';
                element.style.transform = 'translateY(0)';
            }
        });
    }
    
    /**
     * 初始化页面加载动画
     */
    function initPageLoadAnimation() {
        // 为页面元素添加初始动画状态
        const animatedElements = document.querySelectorAll('.category-card, .website-card, .featured-card');
        
        animatedElements.forEach((element, index) => {
            element.style.opacity = '0';
            element.style.transform = 'translateY(30px)';
            element.style.transition = `opacity 0.6s ease ${index * 0.1}s, transform 0.6s ease ${index * 0.1}s`;
        });
        
        // 触发一次滚动动画检查
        animateOnScroll();
    }
    
    // 初始化滚动效果和页面加载动画
    initScrollEffects();
    initPageLoadAnimation();
    
    /**
     * 添加页面访问统计
     */
    function trackPageView() {
        console.log('页面访问统计:', window.location.pathname);
        // 这里可以添加实际的统计代码
    }
    
    // 调用页面访问统计
    trackPageView();
    
    /**
     * 窗口大小改变时的处理
     */
    window.addEventListener('resize', function() {
        // 如果窗口宽度大于768px，关闭移动端菜单
        if (window.innerWidth > 768) {
            menuBtn.classList.remove('active');
            mobileNav.classList.remove('active');
            document.body.style.overflow = '';
        }
    });
});
