/* 项目信息注释 */
/* author: 前端川 */
/* link: www.vsqdc.cn */
/* description: 深耕行业资讯优质导航合集 - JavaScript交互功能 */

// 页面加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化所有交互功能
    initMobileMenu();
    initBannerSlider();
    initSearch();
    initScrollEffects();
});

/**
 * 初始化移动端菜单
 */
function initMobileMenu() {
    const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
    const mobileMenu = document.querySelector('.mobile-menu');
    const mobileNavLinks = document.querySelectorAll('.mobile-nav-link');
    
    if (mobileMenuBtn && mobileMenu) {
        // 点击菜单按钮切换菜单显示/隐藏
        mobileMenuBtn.addEventListener('click', function() {
            mobileMenu.classList.toggle('active');
            // 切换菜单图标样式（如果需要）
            this.classList.toggle('active');
        });
        
        // 点击移动端导航链接后关闭菜单
        mobileNavLinks.forEach(link => {
            link.addEventListener('click', function() {
                mobileMenu.classList.remove('active');
                mobileMenuBtn.classList.remove('active');
            });
        });
        
        // 点击页面其他区域关闭菜单
        document.addEventListener('click', function(e) {
            if (!mobileMenuBtn.contains(e.target) && !mobileMenu.contains(e.target)) {
                mobileMenu.classList.remove('active');
                mobileMenuBtn.classList.remove('active');
            }
        });
    }
}

/**
 * 初始化轮播图
 */
function initBannerSlider() {
    const bannerItems = document.querySelectorAll('.banner-item');
    const indicators = document.querySelectorAll('.indicator');
    const bannerSlider = document.querySelector('.banner-slider');
    
    if (bannerItems.length === 0 || indicators.length === 0) return;
    
    let currentIndex = 0;
    const totalItems = bannerItems.length;
    const autoPlayInterval = 5000; // 自动播放间隔（毫秒）
    let autoPlayTimer = null;
    
    // 显示指定索引的轮播项
    function showSlide(index) {
        // 隐藏所有轮播项
        bannerItems.forEach(item => {
            item.classList.remove('active');
        });
        
        // 移除所有指示器的激活状态
        indicators.forEach(indicator => {
            indicator.classList.remove('active');
        });
        
        // 显示当前轮播项
        bannerItems[index].classList.add('active');
        indicators[index].classList.add('active');
        
        // 更新当前索引
        currentIndex = index;
    }
    
    // 下一张轮播图
    function nextSlide() {
        const nextIndex = (currentIndex + 1) % totalItems;
        showSlide(nextIndex);
    }
    
    // 上一张轮播图
    function prevSlide() {
        const prevIndex = (currentIndex - 1 + totalItems) % totalItems;
        showSlide(prevIndex);
    }
    
    // 点击指示器切换轮播图
    indicators.forEach((indicator, index) => {
        indicator.addEventListener('click', function() {
            showSlide(index);
            // 重置自动播放计时器
            resetAutoPlay();
        });
    });
    
    // 自动播放功能
    function startAutoPlay() {
        autoPlayTimer = setInterval(nextSlide, autoPlayInterval);
    }
    
    // 重置自动播放计时器
    function resetAutoPlay() {
        clearInterval(autoPlayTimer);
        startAutoPlay();
    }
    
    // 鼠标悬停时暂停自动播放
    if (bannerSlider) {
        bannerSlider.addEventListener('mouseenter', function() {
            clearInterval(autoPlayTimer);
        });
        
        // 鼠标离开时恢复自动播放
        bannerSlider.addEventListener('mouseleave', function() {
            startAutoPlay();
        });
    }
    
    // 初始化显示第一张轮播图
    showSlide(0);
    
    // 启动自动播放
    startAutoPlay();
}

/**
 * 初始化搜索功能
 */
function initSearch() {
    const searchBox = document.querySelector('.search-box');
    const searchInput = document.querySelector('.search-input');
    const searchBtn = document.querySelector('.search-btn');
    
    if (searchBox && searchInput && searchBtn) {
        // 点击搜索按钮执行搜索
        searchBtn.addEventListener('click', function() {
            performSearch();
        });
        
        // 按下回车键执行搜索
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch();
            }
        });
        
        // 搜索函数（目前只是简单的console输出）
        function performSearch() {
            const searchTerm = searchInput.value.trim();
            if (searchTerm) {
                console.log('搜索关键词:', searchTerm);
                // 这里可以添加实际的搜索逻辑
                alert('搜索功能开发中，您搜索的关键词是：' + searchTerm);
            }
        }
    }
}

/**
 * 初始化滚动效果
 */
function initScrollEffects() {
    const header = document.querySelector('.header');
    
    // 滚动时添加头部阴影和样式变化
    window.addEventListener('scroll', function() {
        if (window.scrollY > 50) {
            header.classList.add('scrolled');
        } else {
            header.classList.remove('scrolled');
        }
        
        // 滚动触发元素动画
        animateOnScroll();
    });
}

/**
 * 滚动时元素动画
 */
function animateOnScroll() {
    const animatedElements = document.querySelectorAll('.category-item, .website-item, .feature-item');
    
    animatedElements.forEach(element => {
        const elementTop = element.getBoundingClientRect().top;
        const elementVisible = 150;
        
        if (elementTop < window.innerHeight - elementVisible) {
            element.classList.add('animate-in');
        }
    });
}

/**
 * 平滑滚动到锚点
 */
function smoothScrollToAnchor() {
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                e.preventDefault();
                
                // 计算目标位置
                const targetPosition = targetElement.offsetTop - 80; // 减去头部高度
                
                // 平滑滚动
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }
        });
    });
}

/**
 * 添加元素进入视口的动画类
 */
function addAnimationClasses() {
    const animatedElements = document.querySelectorAll('.category-item, .website-item, .feature-item');
    
    animatedElements.forEach((element, index) => {
        element.style.opacity = '0';
        element.style.transform = 'translateY(30px)';
        element.style.transition = `opacity 0.6s ease ${index * 0.1}s, transform 0.6s ease ${index * 0.1}s`;
    });
    
    // 初始检查一次
    animateOnScroll();
}

// 初始化动画类
addAnimationClasses();

/**
 * 导航链接高亮当前页
 */
function highlightCurrentNavLink() {
    const currentUrl = window.location.pathname;
    const navLinks = document.querySelectorAll('.nav-link, .mobile-nav-link');
    
    navLinks.forEach(link => {
        const linkUrl = new URL(link.href).pathname;
        
        if (currentUrl === linkUrl) {
            link.classList.add('active');
        } else {
            link.classList.remove('active');
        }
    });
}

// 初始化导航链接高亮
highlightCurrentNavLink();

/**
 * 图片加载错误处理
 */
function handleImageErrors() {
    const images = document.querySelectorAll('img');
    
    images.forEach(image => {
        image.addEventListener('error', function() {
            // 如果图片加载失败，使用默认图标
            this.src = 'img/default.svg';
            this.alt = '默认图标';
        });
    });
}

// 初始化图片错误处理
handleImageErrors();

/**
 * 添加窗口大小变化时的响应式处理
 */
function initResizeHandler() {
    let resizeTimeout;
    
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            // 窗口大小变化后重新初始化某些功能
            console.log('窗口大小变化，执行响应式处理');
        }, 250); // 防抖，250ms后执行
    });
}

// 初始化窗口大小变化处理
initResizeHandler();

/**
 * 复制功能（可选）
 */
function initCopy功能() {
    // 可以添加复制链接等功能
    const copyButtons = document.querySelectorAll('.copy-btn');
    
    copyButtons.forEach(button => {
        button.addEventListener('click', function() {
            const textToCopy = this.getAttribute('data-copy-text');
            if (textToCopy) {
                navigator.clipboard.writeText(textToCopy)
                    .then(() => {
                        alert('已复制到剪贴板');
                    })
                    .catch(err => {
                        console.error('复制失败:', err);
                    });
            }
        });
    });
}

// 初始化复制功能
initCopy功能();