/**
 * author: 前端川
 * link: www.vsqdc.cn
 * description: 聚焦生活服务导航站点的主JavaScript文件，包含轮播图、移动端菜单等交互功能
 */

// DOM加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化轮播图
    initBannerSlider();
    
    // 初始化移动端菜单
    initMobileMenu();
    
    // 初始化搜索功能
    initSearch();
    
    // 初始化平滑滚动
    initSmoothScroll();
    
    // 初始化页面加载动画
    initPageLoadAnimation();
});

/**
 * 初始化轮播图功能
 */
function initBannerSlider() {
    // 获取轮播相关元素
    const slider = document.querySelector('.banner-slider');
    const items = document.querySelectorAll('.banner-item');
    const indicators = document.querySelectorAll('.indicator');
    
    // 检查元素是否存在
    if (!slider || !items.length || !indicators.length) return;
    
    let currentIndex = 0;
    const totalItems = items.length;
    const intervalTime = 5000; // 轮播间隔时间（毫秒）
    let sliderInterval;
    
    /**
     * 切换到指定索引的轮播项
     * @param {number} index - 目标索引
     */
    function switchToSlide(index) {
        // 移除所有活动状态
        items.forEach(item => item.classList.remove('active'));
        indicators.forEach(indicator => indicator.classList.remove('active'));
        
        // 添加当前活动状态
        items[index].classList.add('active');
        indicators[index].classList.add('active');
        
        // 更新当前索引
        currentIndex = index;
    }
    
    /**
     * 切换到下一张轮播图
     */
    function nextSlide() {
        const nextIndex = (currentIndex + 1) % totalItems;
        switchToSlide(nextIndex);
    }
    
    /**
     * 切换到上一张轮播图
     */
    function prevSlide() {
        const prevIndex = (currentIndex - 1 + totalItems) % totalItems;
        switchToSlide(prevIndex);
    }
    
    // 为指示器添加点击事件
    indicators.forEach((indicator, index) => {
        indicator.addEventListener('click', () => {
            switchToSlide(index);
            // 重置轮播间隔
            resetInterval();
        });
    });
    
    /**
     * 重置轮播间隔
     */
    function resetInterval() {
        clearInterval(sliderInterval);
        sliderInterval = setInterval(nextSlide, intervalTime);
    }
    
    // 自动轮播
    sliderInterval = setInterval(nextSlide, intervalTime);
    
    // 鼠标悬停时暂停轮播
    slider.addEventListener('mouseenter', () => {
        clearInterval(sliderInterval);
    });
    
    // 鼠标离开时恢复轮播
    slider.addEventListener('mouseleave', () => {
        resetInterval();
    });
}

/**
 * 初始化移动端菜单
 */
function initMobileMenu() {
    // 获取菜单相关元素
    const menuBtn = document.getElementById('menuBtn');
    const mobileNav = document.getElementById('mobileNav');
    
    // 检查元素是否存在
    if (!menuBtn || !mobileNav) return;
    
    /**
     * 切换移动端菜单显示状态
     */
    function toggleMobileMenu() {
        mobileNav.classList.toggle('active');
        
        // 切换菜单图标动画效果
        const menuIcon = menuBtn.querySelector('.menu-icon');
        if (menuIcon) {
            menuIcon.style.transition = 'transform 0.3s ease';
            if (mobileNav.classList.contains('active')) {
                menuIcon.style.transform = 'rotate(90deg)';
            } else {
                menuIcon.style.transform = 'rotate(0deg)';
            }
        }
    }
    
    // 菜单按钮点击事件
    menuBtn.addEventListener('click', toggleMobileMenu);
    
    // 移动端导航链接点击事件（点击后关闭菜单）
    const mobileLinks = mobileNav.querySelectorAll('.mobile-nav-link');
    mobileLinks.forEach(link => {
        link.addEventListener('click', () => {
            mobileNav.classList.remove('active');
            const menuIcon = menuBtn.querySelector('.menu-icon');
            if (menuIcon) {
                menuIcon.style.transform = 'rotate(0deg)';
            }
        });
    });
    
    // 点击页面其他区域关闭菜单
    document.addEventListener('click', (e) => {
        if (!mobileNav.contains(e.target) && !menuBtn.contains(e.target)) {
            mobileNav.classList.remove('active');
            const menuIcon = menuBtn.querySelector('.menu-icon');
            if (menuIcon) {
                menuIcon.style.transform = 'rotate(0deg)';
            }
        }
    });
}

/**
 * 初始化搜索功能
 */
function initSearch() {
    // 获取搜索相关元素
    const searchBox = document.querySelector('.search-box');
    const searchInput = document.querySelector('.search-input');
    const searchBtn = document.querySelector('.search-btn');
    
    // 检查元素是否存在
    if (!searchBox || !searchInput || !searchBtn) return;
    
    // 搜索按钮点击事件
    searchBtn.addEventListener('click', () => {
        performSearch();
    });
    
    // 搜索框回车键事件
    searchInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            performSearch();
        }
    });
    
    /**
     * 执行搜索操作
     */
    function performSearch() {
        const searchTerm = searchInput.value.trim();
        if (searchTerm) {
            // 这里可以添加搜索逻辑
            console.log('搜索关键词:', searchTerm);
            // 示例：跳转到搜索结果页面
            // window.location.href = `search.html?q=${encodeURIComponent(searchTerm)}`;
            
            // 简单的搜索反馈
            alert(`搜索功能开发中，您搜索的关键词是：${searchTerm}`);
        }
    }
    
    // 搜索框聚焦效果
    searchInput.addEventListener('focus', () => {
        searchBox.style.boxShadow = '0 0 0 3px rgba(102, 126, 234, 0.3)';
    });
    
    // 搜索框失焦效果
    searchInput.addEventListener('blur', () => {
        searchBox.style.boxShadow = '';
    });
}

/**
 * 初始化平滑滚动
 */
function initSmoothScroll() {
    // 获取所有导航链接
    const navLinks = document.querySelectorAll('a[href^="#"]');
    
    navLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            // 阻止默认跳转行为
            e.preventDefault();
            
            // 获取目标元素
            const targetId = link.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            // 检查目标元素是否存在
            if (targetElement) {
                // 平滑滚动到目标元素
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

/**
 * 初始化页面加载动画
 */
function initPageLoadAnimation() {
    // 获取需要添加动画的元素
    const sections = document.querySelectorAll('section');
    const categoryItems = document.querySelectorAll('.category-item');
    const websiteItems = document.querySelectorAll('.website-item');
    
    // 为元素添加初始动画类
    sections.forEach((section, index) => {
        section.style.opacity = '0';
        section.style.transform = 'translateY(30px)';
        section.style.transition = `opacity 0.8s ease ${index * 0.2}s, transform 0.8s ease ${index * 0.2}s`;
    });
    
    // 页面加载完成后显示元素
    window.addEventListener('load', () => {
        sections.forEach(section => {
            section.style.opacity = '1';
            section.style.transform = 'translateY(0)';
        });
    });
    
    // 为分类项添加悬停动画效果
    categoryItems.forEach(item => {
        item.addEventListener('mouseenter', () => {
            item.style.transform = 'translateY(-10px) scale(1.02)';
        });
        
        item.addEventListener('mouseleave', () => {
            item.style.transform = 'translateY(0) scale(1)';
        });
    });
    
    // 为网站项添加悬停动画效果
    websiteItems.forEach(item => {
        item.addEventListener('mouseenter', () => {
            item.style.transform = 'translateY(-5px) scale(1.01)';
        });
        
        item.addEventListener('mouseleave', () => {
            item.style.transform = 'translateY(0) scale(1)';
        });
    });
}

/**
 * 初始化分类页面的筛选功能（如果是分类页面）
 */
function initCategoryFilter() {
    // 检查是否为分类页面
    if (!document.querySelector('.category-page')) return;
    
    // 获取筛选相关元素
    const filterButtons = document.querySelectorAll('.filter-btn');
    const websiteItems = document.querySelectorAll('.website-item');
    
    // 检查元素是否存在
    if (!filterButtons.length || !websiteItems.length) return;
    
    // 筛选按钮点击事件
    filterButtons.forEach(button => {
        button.addEventListener('click', () => {
            // 移除所有按钮的活动状态
            filterButtons.forEach(btn => btn.classList.remove('active'));
            // 添加当前按钮的活动状态
            button.classList.add('active');
            
            // 获取筛选类别
            const filterCategory = button.getAttribute('data-category');
            
            // 筛选网站项
            websiteItems.forEach(item => {
                if (filterCategory === 'all' || item.getAttribute('data-category') === filterCategory) {
                    item.style.display = 'flex';
                    item.style.animation = 'fadeInUp 0.5s ease';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    });
}

/**
 * 处理页面滚动事件
 */
window.addEventListener('scroll', () => {
    // 获取页面头部
    const header = document.querySelector('.header');
    
    // 检查元素是否存在
    if (!header) return;
    
    // 页面滚动时添加/移除头部样式
    if (window.scrollY > 50) {
        header.style.background = 'linear-gradient(135deg, rgba(102, 126, 234, 0.95) 0%, rgba(118, 75, 162, 0.95) 100%)';
        header.style.boxShadow = '0 4px 20px rgba(0, 0, 0, 0.15)';
        header.style.padding = '10px 20px';
    } else {
        header.style.background = 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
        header.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
        header.style.padding = '15px 20px';
    }
});

/**
 * 初始化快速链接交互效果
 */
function initQuickLinks() {
    // 获取快速链接元素
    const quickLinks = document.querySelectorAll('.quick-link');
    
    // 检查元素是否存在
    if (!quickLinks.length) return;
    
    quickLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            // 添加点击动画效果
            link.style.transform = 'scale(0.95)';
            setTimeout(() => {
                link.style.transform = '';
            }, 150);
        });
    });
}

// 初始化快速链接交互效果
document.addEventListener('DOMContentLoaded', initQuickLinks);

/**
 * 动态生成网站项（示例数据）
 */
function generateWebsiteItems() {
    // 检查是否需要生成网站项
    const websiteGrid = document.querySelector('.website-grid');
    if (!websiteGrid || websiteGrid.children.length > 0) return;
    
    // 示例网站数据
    const websites = [
        {
            name: '美团外卖',
            description: '便捷的在线外卖服务平台',
            logo: 'img/website1.svg',
            category: 'food'
        },
        {
            name: '携程旅行',
            description: '全面的旅游出行服务平台',
            logo: 'img/website2.svg',
            category: 'travel'
        },
        {
            name: '淘宝',
            description: '中国领先的在线购物平台',
            logo: 'img/website3.svg',
            category: 'shopping'
        },
        {
            name: '京东',
            description: '综合网上购物商城',
            logo: 'img/website4.svg',
            category: 'shopping'
        },
        {
            name: '百度地图',
            description: '精准的地图导航服务',
            logo: 'img/website5.svg',
            category: 'travel'
        },
        {
            name: '腾讯视频',
            description: '领先的在线视频平台',
            logo: 'img/website6.svg',
            category: 'entertainment'
        }
    ];
    
    // 生成网站项
    websites.forEach(website => {
        const websiteItem = document.createElement('a');
        websiteItem.className = 'website-item';
        websiteItem.href = '#';
        websiteItem.setAttribute('data-category', website.category);
        
        websiteItem.innerHTML = `
            <div class="website-logo">
                <img src="${website.logo}" alt="${website.name}">
            </div>
            <div class="website-info">
                <h3>${website.name}</h3>
                <p>${website.description}</p>
            </div>
        `;
        
        websiteGrid.appendChild(websiteItem);
    });
}

// 初始化动态生成网站项
document.addEventListener('DOMContentLoaded', generateWebsiteItems);