/**
 * 极简风格清爽整洁导航页JavaScript文件
 * @author 前端川
 * @link www.vsqdc.cn
 * @description 包含页面交互逻辑和搜索功能
 */

// DOM加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 移动端菜单切换功能
    initMobileMenu();
    
    // 搜索功能
    initSearch();
    
    // 导航栏滚动效果
    initNavScroll();
    
    // 网站卡片悬停效果增强
    initSiteCardEffects();
});

/**
 * 初始化移动端菜单切换功能
 */
function initMobileMenu() {
    const menuToggle = document.getElementById('menuToggle');
    const nav = document.querySelector('.nav');
    
    if (menuToggle && nav) {
        menuToggle.addEventListener('click', function() {
            // 切换菜单按钮状态
            this.classList.toggle('active');
            // 切换导航菜单显示/隐藏
            nav.classList.toggle('active');
        });
        
        // 点击导航链接后关闭菜单
        const navLinks = document.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            link.addEventListener('click', function() {
                menuToggle.classList.remove('active');
                nav.classList.remove('active');
            });
        });
    }
}

/**
 * 初始化搜索功能
 */
function initSearch() {
    const searchInput = document.getElementById('searchInput');
    const searchBtn = document.querySelector('.search-btn');
    
    if (searchInput && searchBtn) {
        // 搜索按钮点击事件
        searchBtn.addEventListener('click', function() {
            performSearch();
        });
        
        // 回车键搜索
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch();
            }
        });
        
        // 搜索输入框焦点效果
        searchInput.addEventListener('focus', function() {
            this.parentElement.style.transform = 'translateY(-2px)';
            this.parentElement.style.boxShadow = '0 15px 50px rgba(0, 0, 0, 0.2)';
        });
        
        searchInput.addEventListener('blur', function() {
            this.parentElement.style.transform = 'translateY(0)';
            this.parentElement.style.boxShadow = '0 10px 40px rgba(0, 0, 0, 0.15)';
        });
    }
}

/**
 * 执行搜索操作
 */
function performSearch() {
    const searchInput = document.getElementById('searchInput');
    const query = searchInput.value.trim();
    
    if (query) {
        // 默认使用百度搜索
        window.open(`https://www.baidu.com/s?wd=${encodeURIComponent(query)}`, '_blank');
        // 清空搜索框
        searchInput.value = '';
    }
}

/**
 * 初始化导航栏滚动效果
 */
function initNavScroll() {
    const header = document.querySelector('.header');
    
    if (header) {
        window.addEventListener('scroll', function() {
            if (window.scrollY > 50) {
                header.style.background = 'rgba(255, 255, 255, 0.98)';
                header.style.boxShadow = '0 4px 30px rgba(0, 0, 0, 0.15)';
            } else {
                header.style.background = 'rgba(255, 255, 255, 0.95)';
                header.style.boxShadow = '0 2px 20px rgba(0, 0, 0, 0.1)';
            }
        });
    }
}

/**
 * 初始化网站卡片悬停效果增强
 */
function initSiteCardEffects() {
    const siteCards = document.querySelectorAll('.site-card');
    
    siteCards.forEach(card => {
        // 鼠标进入效果
        card.addEventListener('mouseenter', function(e) {
            // 添加轻微的缩放和旋转效果
            this.style.transition = 'all 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
        });
        
        // 鼠标离开效果
        card.addEventListener('mouseleave', function(e) {
            this.style.transition = 'all 0.3s ease';
        });
        
        // 点击效果
        card.addEventListener('click', function(e) {
            // 添加点击波纹效果
            createRipple(e, this);
        });
    });
}

/**
 * 创建点击波纹效果
 * @param {MouseEvent} e - 点击事件对象
 * @param {HTMLElement} element - 要添加波纹效果的元素
 */
function createRipple(e, element) {
    // 创建波纹元素
    const ripple = document.createElement('span');
    
    // 获取元素位置和尺寸
    const rect = element.getBoundingClientRect();
    const size = Math.max(rect.width, rect.height);
    const x = e.clientX - rect.left - size / 2;
    const y = e.clientY - rect.top - size / 2;
    
    // 设置波纹样式
    ripple.style.cssText = `
        position: absolute;
        border-radius: 50%;
        background: rgba(102, 126, 234, 0.3);
        transform: scale(0);
        animation: ripple 0.6s linear;
        width: ${size}px;
        height: ${size}px;
        left: ${x}px;
        top: ${y}px;
        pointer-events: none;
    `;
    
    // 添加波纹到元素
    element.style.position = 'relative';
    element.style.overflow = 'hidden';
    element.appendChild(ripple);
    
    // 动画结束后移除波纹
    setTimeout(() => {
        ripple.remove();
    }, 600);
}

// 添加波纹动画
const style = document.createElement('style');
style.textContent = `
    @keyframes ripple {
        to {
            transform: scale(4);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

/**
 * 平滑滚动到锚点
 * @param {string} selector - 要滚动到的元素选择器
 */
function smoothScroll(selector) {
    const element = document.querySelector(selector);
    if (element) {
        element.scrollIntoView({
            behavior: 'smooth',
            block: 'start'
        });
    }
}

/**
 * 获取当前页面路径
 * @returns {string} 当前页面路径
 */
function getCurrentPage() {
    return window.location.pathname.split('/').pop() || 'index.html';
}

/**
 * 设置导航栏活动状态
 */
function setActiveNavLink() {
    const currentPage = getCurrentPage();
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        const linkPage = link.getAttribute('href').split('/').pop() || 'index.html';
        if (linkPage === currentPage) {
            link.classList.add('active');
        } else {
            link.classList.remove('active');
        }
    });
}

// 页面加载完成后设置活动导航链接
window.addEventListener('load', function() {
    setActiveNavLink();
});

// 窗口大小改变时重新初始化一些功能
window.addEventListener('resize', function() {
    // 确保导航菜单在大屏幕上始终显示
    const nav = document.querySelector('.nav');
    const menuToggle = document.getElementById('menuToggle');
    
    if (window.innerWidth > 768) {
        nav.classList.remove('active');
        if (menuToggle) {
            menuToggle.classList.remove('active');
        }
    }
});