/*
 * 职场导航JavaScript功能文件
 * @author: 前端川
 * @link: www.vsqdc.cn
 * @description: 职场导航网站的主JavaScript文件，实现各种交互功能
 */

// 页面加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化移动端菜单
    initMobileMenu();
    
    // 初始化搜索功能
    initSearch();
    
    // 初始化收藏功能
    initFavorites();
    
    // 初始化平滑滚动
    initSmoothScroll();
    
    // 初始化页面加载动画
    initPageLoad();
    
    // 初始化收藏列表显示
    initFavoritesList();
});

/**
 * 初始化移动端菜单
 * 实现移动端菜单的显示和隐藏功能
 */
function initMobileMenu() {
    const menuBtn = document.getElementById('menuBtn');
    const mobileNav = document.getElementById('mobileNav');
    
    if (menuBtn && mobileNav) {
        menuBtn.addEventListener('click', function() {
            mobileNav.classList.toggle('active');
            
            // 切换菜单按钮的动画效果
            const menuIcons = menuBtn.querySelectorAll('.menu-icon');
            menuIcons[0].classList.toggle('rotate-45');
            menuIcons[1].classList.toggle('opacity-0');
            menuIcons[2].classList.toggle('rotate--45');
        });
        
        // 点击移动端菜单外部关闭菜单
        document.addEventListener('click', function(e) {
            if (!menuBtn.contains(e.target) && !mobileNav.contains(e.target)) {
                mobileNav.classList.remove('active');
                
                // 恢复菜单按钮的初始状态
                const menuIcons = menuBtn.querySelectorAll('.menu-icon');
                menuIcons[0].classList.remove('rotate-45');
                menuIcons[1].classList.remove('opacity-0');
                menuIcons[2].classList.remove('rotate--45');
            }
        });
    }
}

/**
 * 初始化搜索功能
 * 实现网站搜索功能
 */
function initSearch() {
    const searchInput = document.getElementById('searchInput');
    const searchBtn = document.getElementById('searchBtn');
    
    if (searchInput && searchBtn) {
        // 点击搜索按钮执行搜索
        searchBtn.addEventListener('click', function() {
            performSearch();
        });
        
        // 按下回车键执行搜索
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch();
            }
        });
    }
}

/**
 * 执行搜索操作
 */
function performSearch() {
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        const searchTerm = searchInput.value.trim();
        if (searchTerm) {
            // 这里可以添加实际的搜索逻辑
            console.log('搜索关键词：', searchTerm);
            alert(`正在搜索 "${searchTerm}"，搜索功能正在开发中...`);
        } else {
            alert('请输入搜索关键词');
        }
    }
}

/**
 * 初始化收藏功能
 * 实现网站收藏和取消收藏功能
 */
function initFavorites() {
    const favoriteBtns = document.querySelectorAll('.website-fav');
    
    favoriteBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            // 切换收藏状态
            this.classList.toggle('active');
            
            // 获取网站信息
            const websiteItem = this.closest('.website-item');
            const websiteName = websiteItem.querySelector('.website-name').textContent;
            
            if (this.classList.contains('active')) {
                // 添加到收藏
                addToFavorites(websiteName);
                alert(`已将 "${websiteName}" 添加到收藏`);
            } else {
                // 从收藏中移除
                removeFromFavorites(websiteName);
                alert(`已将 "${websiteName}" 从收藏中移除`);
            }
            
            // 如果在收藏页面，更新收藏列表
            if (window.location.pathname.includes('favorites.html')) {
                initFavoritesList();
            }
        });
    });
}

/**
 * 添加到收藏
 * @param {string} websiteName - 网站名称
 */
function addToFavorites(websiteName) {
    // 从本地存储获取收藏列表
    let favorites = JSON.parse(localStorage.getItem('favorites')) || [];
    
    // 检查是否已收藏
    if (!favorites.includes(websiteName)) {
        favorites.push(websiteName);
        localStorage.setItem('favorites', JSON.stringify(favorites));
    }
}

/**
 * 从收藏中移除
 * @param {string} websiteName - 网站名称
 */
function removeFromFavorites(websiteName) {
    // 从本地存储获取收藏列表
    let favorites = JSON.parse(localStorage.getItem('favorites')) || [];
    
    // 过滤掉要移除的网站
    favorites = favorites.filter(name => name !== websiteName);
    localStorage.setItem('favorites', JSON.stringify(favorites));
}

/**
 * 初始化收藏列表显示
 * 在收藏页面中显示用户的收藏列表
 */
function initFavoritesList() {
    // 检查是否在收藏页面
    if (window.location.pathname.includes('favorites.html')) {
        const emptyFavorites = document.getElementById('emptyFavorites');
        const favoritesList = document.getElementById('favoritesList');
        
        // 从本地存储获取收藏列表
        const favorites = JSON.parse(localStorage.getItem('favorites')) || [];
        
        if (favorites.length > 0) {
            // 显示收藏列表
            emptyFavorites.style.display = 'none';
            favoritesList.style.display = 'block';
        } else {
            // 显示空收藏提示
            emptyFavorites.style.display = 'block';
            favoritesList.style.display = 'none';
        }
    }
}

/**
 * 初始化平滑滚动
 * 实现页面内锚点的平滑滚动效果
 */
function initSmoothScroll() {
    // 为所有锚点链接添加平滑滚动效果
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            // 检查是否是本页锚点
            if (this.getAttribute('href') === '#') {
                e.preventDefault();
                return;
            }
            
            // 检查目标元素是否存在
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                e.preventDefault();
                
                // 计算滚动位置（考虑固定头部的高度）
                const headerHeight = document.querySelector('.header').offsetHeight;
                const targetPosition = targetElement.offsetTop - headerHeight;
                
                // 执行平滑滚动
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }
        });
    });
}

/**
 * 初始化页面加载动画
 * 实现页面加载时的过渡效果
 */
function initPageLoad() {
    // 添加页面加载完成的类
    document.body.classList.add('page-loaded');
    
    // 为页面元素添加进入动画
    const animateElements = document.querySelectorAll('.card, .category-item, .quick-link, .website-item, .contact-item');
    
    animateElements.forEach((element, index) => {
        // 为每个元素设置延迟动画
        element.style.animationDelay = `${index * 0.1}s`;
    });
}

/**
 * 显示加载动画
 * @param {HTMLElement} element - 要添加加载动画的元素
 */
function showLoading(element) {
    if (element) {
        element.classList.add('loading');
    }
}

/**
 * 隐藏加载动画
 * @param {HTMLElement} element - 要移除加载动画的元素
 */
function hideLoading(element) {
    if (element) {
        element.classList.remove('loading');
    }
}

/**
 * 获取URL参数
 * @param {string} name - 参数名称
 * @returns {string|null} - 参数值
 */
function getUrlParam(name) {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get(name);
}

/**
 * 设置URL参数
 * @param {string} name - 参数名称
 * @param {string} value - 参数值
 */
function setUrlParam(name, value) {
    const urlParams = new URLSearchParams(window.location.search);
    urlParams.set(name, value);
    window.history.replaceState({}, '', `${window.location.pathname}?${urlParams}`);
}

/**
 * 移除URL参数
 * @param {string} name - 参数名称
 */
function removeUrlParam(name) {
    const urlParams = new URLSearchParams(window.location.search);
    urlParams.delete(name);
    window.history.replaceState({}, '', `${window.location.pathname}${urlParams.toString() ? '?' + urlParams : ''}`);
}

/**
 * 防抖函数
 * @param {Function} func - 要执行的函数
 * @param {number} delay - 延迟时间（毫秒）
 * @returns {Function} - 防抖处理后的函数
 */
function debounce(func, delay) {
    let timeoutId;
    return function(...args) {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => func.apply(this, args), delay);
    };
}

/**
 * 节流函数
 * @param {Function} func - 要执行的函数
 * @param {number} limit - 时间限制（毫秒）
 * @returns {Function} - 节流处理后的函数
 */
function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

/**
 * 动态添加CSS样式
 * @param {string} css - CSS样式字符串
 */
function addCss(css) {
    const style = document.createElement('style');
    style.textContent = css;
    document.head.appendChild(style);
}

/**
 * 动态移除CSS样式
 * @param {string} selector - 要移除的CSS选择器
 */
function removeCss(selector) {
    const style = document.createElement('style');
    style.textContent = `${selector} { display: none !important; }`;
    document.head.appendChild(style);
}

/**
 * 检测设备类型
 * @returns {string} - 设备类型（mobile或desktop）
 */
function detectDevice() {
    return window.innerWidth < 768 ? 'mobile' : 'desktop';
}

/**
 * 检测浏览器类型
 * @returns {string} - 浏览器类型
 */
function detectBrowser() {
    const userAgent = navigator.userAgent;
    if (userAgent.includes('Chrome')) return 'chrome';
    if (userAgent.includes('Firefox')) return 'firefox';
    if (userAgent.includes('Safari')) return 'safari';
    if (userAgent.includes('Edge')) return 'edge';
    if (userAgent.includes('IE')) return 'ie';
    return 'unknown';
}

/**
 * 复制文本到剪贴板
 * @param {string} text - 要复制的文本
 * @returns {Promise<boolean>} - 是否复制成功
 */
async function copyToClipboard(text) {
    try {
        await navigator.clipboard.writeText(text);
        return true;
    } catch (err) {
        console.error('复制失败：', err);
        return false;
    }
}

/**
 * 获取本地存储数据
 * @param {string} key - 存储键名
 * @param {any} defaultValue - 默认值
 * @returns {any} - 存储值
 */
function getLocalStorage(key, defaultValue = null) {
    try {
        const value = localStorage.getItem(key);
        return value ? JSON.parse(value) : defaultValue;
    } catch (err) {
        console.error('获取本地存储失败：', err);
        return defaultValue;
    }
}

/**
 * 设置本地存储数据
 * @param {string} key - 存储键名
 * @param {any} value - 存储值
 * @returns {boolean} - 是否设置成功
 */
function setLocalStorage(key, value) {
    try {
        localStorage.setItem(key, JSON.stringify(value));
        return true;
    } catch (err) {
        console.error('设置本地存储失败：', err);
        return false;
    }
}

/**
 * 删除本地存储数据
 * @param {string} key - 存储键名
 * @returns {boolean} - 是否删除成功
 */
function removeLocalStorage(key) {
    try {
        localStorage.removeItem(key);
        return true;
    } catch (err) {
        console.error('删除本地存储失败：', err);
        return false;
    }
}

/**
 * 清空本地存储数据
 * @returns {boolean} - 是否清空成功
 */
function clearLocalStorage() {
    try {
        localStorage.clear();
        return true;
    } catch (err) {
        console.error('清空本地存储失败：', err);
        return false;
    }
}