/*
 * author: 前端川
 * link: www.vsqdc.cn
 * description: 学生学习导航站JavaScript文件，包含各种交互功能
 */

// 文档加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化所有功能
    initMobileMenu();
    initScrollAnimations();
    initSearchFunctionality();
    initSmoothScroll();
    initCardHoverEffects();
    initNavbarScroll();
});

/**
 * 初始化移动端菜单功能
 */
function initMobileMenu() {
    const menuBtn = document.getElementById('menuBtn');
    const navMenu = document.querySelector('.nav-menu');
    
    if (menuBtn && navMenu) {
        menuBtn.addEventListener('click', function() {
            // 切换菜单按钮状态
            this.classList.toggle('active');
            // 切换导航菜单显示状态
            navMenu.classList.toggle('active');
        });
        
        // 点击导航链接后关闭菜单
        const navLinks = navMenu.querySelectorAll('a');
        navLinks.forEach(link => {
            link.addEventListener('click', function() {
                menuBtn.classList.remove('active');
                navMenu.classList.remove('active');
            });
        });
    }
}

/**
 * 初始化滚动动画效果
 */
function initScrollAnimations() {
    const fadeElements = document.querySelectorAll('.fade-in');
    
    // 检查元素是否在视口中
    function isInViewport(element) {
        const rect = element.getBoundingClientRect();
        return (
            rect.top <= (window.innerHeight || document.documentElement.clientHeight) * 0.8 &&
            rect.bottom >= 0
        );
    }
    
    // 显示在视口中的元素
    function showVisibleElements() {
        fadeElements.forEach(element => {
            if (isInViewport(element)) {
                element.classList.add('visible');
            }
        });
    }
    
    // 初始检查
    showVisibleElements();
    
    // 滚动时检查
    window.addEventListener('scroll', showVisibleElements);
}

/**
 * 初始化搜索功能
 */
function initSearchFunctionality() {
    const searchBox = document.querySelector('.search-box');
    if (searchBox) {
        const searchInput = searchBox.querySelector('input');
        const searchButton = searchBox.querySelector('button');
        
        // 搜索按钮点击事件
        searchButton.addEventListener('click', function() {
            performSearch(searchInput.value);
        });
        
        // 回车键搜索
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch(searchInput.value);
            }
        });
    }
}

/**
 * 执行搜索操作
 * @param {string} query - 搜索关键词
 */
function performSearch(query) {
    if (query.trim() === '') {
        alert('请输入搜索关键词');
        return;
    }
    
    // 这里可以添加实际的搜索逻辑
    console.log('搜索关键词:', query);
    alert(`正在搜索: "${query}"\n搜索功能正在开发中...`);
}

/**
 * 初始化平滑滚动
 */
function initSmoothScroll() {
    // 为所有锚点链接添加平滑滚动
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                window.scrollTo({
                    top: targetElement.offsetTop - 80, // 考虑导航栏高度
                    behavior: 'smooth'
                });
            }
        });
    });
}

/**
 * 初始化卡片悬停效果
 */
function initCardHoverEffects() {
    const cards = document.querySelectorAll('.category-card, .recommendation-card');
    
    cards.forEach(card => {
        // 添加鼠标进入事件
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-10px) rotateX(5deg) scale(1.02)';
        });
        
        // 添加鼠标离开事件
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) rotateX(0) scale(1)';
        });
    });
}

/**
 * 初始化导航栏滚动效果
 */
function initNavbarScroll() {
    const navbar = document.querySelector('.navbar');
    if (!navbar) return;
    
    let lastScrollTop = 0;
    const navbarHeight = navbar.offsetHeight;
    
    window.addEventListener('scroll', function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        if (scrollTop > lastScrollTop && scrollTop > navbarHeight) {
            // 向下滚动且超过导航栏高度，隐藏导航栏
            navbar.style.transform = 'translateY(-100%)';
        } else {
            // 向上滚动或未超过导航栏高度，显示导航栏
            navbar.style.transform = 'translateY(0)';
        }
        
        // 如果滚动到顶部，恢复导航栏样式
        if (scrollTop === 0) {
            navbar.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
        } else {
            navbar.style.boxShadow = '0 4px 20px rgba(0, 0, 0, 0.15)';
        }
        
        lastScrollTop = scrollTop;
    });
}

/**
 * 随机生成一个范围内的数字
 * @param {number} min - 最小值
 * @param {number} max - 最大值
 * @returns {number} 随机数字
 */
function getRandomNumber(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
}

/**
 * 添加页面加载动画
 */
window.addEventListener('load', function() {
    // 移除加载动画（如果有的话）
    const loader = document.querySelector('.loader');
    if (loader) {
        loader.style.display = 'none';
    }
    
    // 页面加载完成后添加淡入效果
    document.body.style.opacity = '0';
    document.body.style.transition = 'opacity 0.5s ease';
    setTimeout(() => {
        document.body.style.opacity = '1';
    }, 100);
});

/**
 * 初始化图片懒加载（如果有需要）
 */
function initLazyLoading() {
    // 检查浏览器是否支持Intersection Observer
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const image = entry.target;
                    image.src = image.dataset.src;
                    image.classList.remove('lazy');
                    imageObserver.unobserve(image);
                }
            });
        });
        
        const lazyImages = document.querySelectorAll('img.lazy');
        lazyImages.forEach(image => {
            imageObserver.observe(image);
        });
    }
}

/**
 * 初始化计数器动画（如果有需要）
 */
function initCounterAnimations() {
    const counters = document.querySelectorAll('.counter');
    let animationStarted = false;
    
    function animateCounter(counter) {
        const target = parseInt(counter.innerText);
        const duration = 2000;
        const step = target / (duration / 16);
        let current = 0;
        
        const timer = setInterval(() => {
            current += step;
            if (current >= target) {
                counter.innerText = target;
                clearInterval(timer);
            } else {
                counter.innerText = Math.floor(current);
            }
        }, 16);
    }
    
    function checkCounters() {
        if (animationStarted) return;
        
        counters.forEach(counter => {
            if (isInViewport(counter)) {
                animateCounter(counter);
                animationStarted = true;
            }
        });
    }
    
    window.addEventListener('scroll', checkCounters);
    checkCounters(); // 初始检查
}

/**
 * 工具函数：防抖
 * @param {Function} func - 要执行的函数
 * @param {number} delay - 延迟时间（毫秒）
 * @returns {Function} 防抖处理后的函数
 */
function debounce(func, delay) {
    let timeoutId;
    return function(...args) {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => func.apply(this, args), delay);
    };
}

/**
 * 工具函数：节流
 * @param {Function} func - 要执行的函数
 * @param {number} limit - 时间限制（毫秒）
 * @returns {Function} 节流处理后的函数
 */
function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

/**
 * 初始化主题切换功能（如果有需要）
 */
function initThemeToggle() {
    const themeToggle = document.querySelector('.theme-toggle');
    if (themeToggle) {
        themeToggle.addEventListener('click', function() {
            document.body.classList.toggle('dark-theme');
            
            // 保存主题偏好到localStorage
            const currentTheme = document.body.classList.contains('dark-theme') ? 'dark' : 'light';
            localStorage.setItem('theme', currentTheme);
        });
        
        // 从localStorage加载主题偏好
        const savedTheme = localStorage.getItem('theme');
        if (savedTheme === 'dark') {
            document.body.classList.add('dark-theme');
        }
    }
}

/**
 * 初始化分享功能（如果有需要）
 */
function initShareFunctionality() {
    const shareButtons = document.querySelectorAll('.share-btn');
    
    shareButtons.forEach(button => {
        button.addEventListener('click', function() {
            const url = window.location.href;
            const title = document.title;
            
            // 检查浏览器是否支持Web Share API
            if (navigator.share) {
                navigator.share({
                    title: title,
                    url: url
                }).catch(err => {
                    console.error('分享失败:', err);
                    fallbackShare(url, title);
                });
            } else {
                // 降级方案：复制链接到剪贴板
                fallbackShare(url, title);
            }
        });
    });
    
    function fallbackShare(url, title) {
        // 创建临时输入框复制链接
        const tempInput = document.createElement('input');
        tempInput.value = url;
        document.body.appendChild(tempInput);
        tempInput.select();
        document.execCommand('copy');
        document.body.removeChild(tempInput);
        
        alert('链接已复制到剪贴板:\n' + url);
    }
}

/**
 * 初始化页面切换动画
 */
function initPageTransition() {
    const links = document.querySelectorAll('a:not([target="_blank"])');
    
    links.forEach(link => {
        link.addEventListener('click', function(e) {
            const target = this.getAttribute('href');
            
            // 只对内部链接应用页面切换动画
            if (target.startsWith('#') || target.startsWith('javascript:')) {
                return;
            }
            
            e.preventDefault();
            
            // 页面淡出动画
            document.body.style.opacity = '0';
            document.body.style.transition = 'opacity 0.3s ease';
            
            setTimeout(() => {
                window.location.href = target;
            }, 300);
        });
    });
}

// 暴露一些函数到全局，方便在HTML中直接调用
window.initMobileMenu = initMobileMenu;
window.initScrollAnimations = initScrollAnimations;
window.performSearch = performSearch;