/*
* author: 前端川
* link: www.vsqdc.cn
* description: 小众特色内容聚合导航页JavaScript文件
*/

// DOM加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化所有功能
    initMobileMenu();
    initSearch();
    initCategoryFilters();
    initSmoothScroll();
    initAnimations();
    initCardInteractions();
});

/**
 * 初始化移动端菜单
 */
function initMobileMenu() {
    const menuBtn = document.getElementById('menuBtn');
    const nav = document.querySelector('.nav');
    
    if (menuBtn && nav) {
        menuBtn.addEventListener('click', function() {
            // 切换导航菜单显示状态
            nav.classList.toggle('active');
            
            // 切换菜单按钮图标样式
            this.classList.toggle('active');
        });
        
        // 点击导航链接后关闭菜单
        const navLinks = nav.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            link.addEventListener('click', function() {
                nav.classList.remove('active');
                menuBtn.classList.remove('active');
            });
        });
        
        // 点击页面其他区域关闭菜单
        document.addEventListener('click', function(e) {
            if (!menuBtn.contains(e.target) && !nav.contains(e.target)) {
                nav.classList.remove('active');
                menuBtn.classList.remove('active');
            }
        });
    }
}

/**
 * 初始化搜索功能
 */
function initSearch() {
    const searchForm = document.querySelector('.search-form');
    const searchInput = document.querySelector('.search-input');
    const searchBtn = document.querySelector('.search-btn');
    
    if (searchForm && searchInput && searchBtn) {
        // 搜索按钮点击事件
        searchBtn.addEventListener('click', function() {
            const searchTerm = searchInput.value.trim();
            if (searchTerm) {
                performSearch(searchTerm);
            }
        });
        
        // 回车键搜索
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                const searchTerm = searchInput.value.trim();
                if (searchTerm) {
                    performSearch(searchTerm);
                }
            }
        });
        
        // 搜索输入框焦点效果
        searchInput.addEventListener('focus', function() {
            searchForm.style.boxShadow = '0 15px 40px rgba(0, 0, 0, 0.3)';
        });
        
        searchInput.addEventListener('blur', function() {
            searchForm.style.boxShadow = '0 10px 30px rgba(0, 0, 0, 0.2)';
        });
    }
}

/**
 * 执行搜索操作
 * @param {string} searchTerm - 搜索关键词
 */
function performSearch(searchTerm) {
    console.log('搜索关键词:', searchTerm);
    // 这里可以添加实际的搜索逻辑
    // 例如：跳转到搜索结果页面或过滤当前页面内容
    alert(`正在搜索: ${searchTerm}`);
}

/**
 * 初始化分类筛选功能
 */
function initCategoryFilters() {
    const filterBtns = document.querySelectorAll('.filter-btn');
    
    if (filterBtns.length > 0) {
        filterBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                // 移除所有按钮的active类
                filterBtns.forEach(b => b.classList.remove('active'));
                // 为当前点击的按钮添加active类
                this.classList.add('active');
                
                // 获取筛选类别
                const category = this.dataset.category || 'all';
                // 执行筛选逻辑
                filterContent(category);
            });
        });
    }
}

/**
 * 根据分类筛选内容
 * @param {string} category - 分类名称
 */
function filterContent(category) {
    const contentCards = document.querySelectorAll('.content-card');
    
    if (contentCards.length > 0) {
        contentCards.forEach(card => {
            if (category === 'all' || card.dataset.category === category) {
                // 显示匹配的卡片
                card.style.display = 'block';
                // 添加动画效果
                card.classList.add('fade-in-up');
                // 动画结束后移除类，以便下次动画可以再次触发
                setTimeout(() => {
                    card.classList.remove('fade-in-up');
                }, 600);
            } else {
                // 隐藏不匹配的卡片
                card.style.display = 'none';
            }
        });
    }
}

/**
 * 初始化平滑滚动
 */
function initSmoothScroll() {
    // 为所有锚点链接添加平滑滚动
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                const offsetTop = targetElement.offsetTop - 80; // 考虑导航栏高度
                
                // 平滑滚动到目标位置
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    });
}

/**
 * 初始化动画效果
 */
function initAnimations() {
    // 监听滚动事件，为进入视口的元素添加动画
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('fade-in-up');
                // 动画触发后停止观察
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);
    
    // 观察所有需要动画的元素
    const animateElements = document.querySelectorAll('.category-card, .recommend-card, .content-card, .feature-item');
    animateElements.forEach(el => {
        observer.observe(el);
    });
}

/**
 * 初始化卡片交互效果
 */
function initCardInteractions() {
    // 分类卡片点击事件
    const categoryCards = document.querySelectorAll('.category-card');
    categoryCards.forEach(card => {
        card.addEventListener('click', function() {
            const categoryName = this.querySelector('.category-name').textContent;
            console.log('点击分类:', categoryName);
            // 这里可以添加跳转到对应分类页面的逻辑
            // window.location.href = `category.html?type=${encodeURIComponent(categoryName)}`;
        });
    });
    
    // 推荐卡片悬停效果增强
    const recommendCards = document.querySelectorAll('.recommend-card');
    recommendCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px) scale(1.02)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
    
    // 内容卡片访问按钮点击事件
    const visitBtns = document.querySelectorAll('.visit-btn');
    visitBtns.forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            const card = this.closest('.content-card');
            const title = card.querySelector('.content-title').textContent;
            console.log('访问内容:', title);
            // 这里可以添加实际的访问逻辑
            alert(`正在访问: ${title}`);
        });
    });
}

/**
 * 初始化导航激活状态
 */
function initNavActiveState() {
    const currentUrl = window.location.pathname;
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        const linkUrl = new URL(link.href).pathname;
        
        // 比较当前URL和链接URL，设置激活状态
        if (currentUrl === linkUrl || 
            (currentUrl.endsWith('/') && linkUrl === currentUrl + 'index.html')) {
            link.classList.add('active');
        } else {
            link.classList.remove('active');
        }
    });
}

/**
 * 添加页面加载动画
 */
function initPageLoadAnimation() {
    // 页面加载完成后移除加载动画
    window.addEventListener('load', function() {
        const loader = document.getElementById('page-loader');
        if (loader) {
            loader.style.opacity = '0';
            loader.style.visibility = 'hidden';
        }
    });
}

/**
 * 初始化响应式调整
 */
function initResponsiveAdjustments() {
    // 监听窗口大小变化，执行相应调整
    window.addEventListener('resize', function() {
        // 确保在窗口变大时关闭移动端菜单
        const nav = document.querySelector('.nav');
        const menuBtn = document.getElementById('menuBtn');
        
        if (window.innerWidth > 768) {
            nav.classList.remove('active');
            menuBtn.classList.remove('active');
        }
    });
}

/**
 * 防抖函数
 * @param {Function} func - 要执行的函数
 * @param {number} delay - 延迟时间（毫秒）
 * @returns {Function} - 防抖处理后的函数
 */
function debounce(func, delay) {
    let timeoutId;
    return function(...args) {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => func.apply(this, args), delay);
    };
}

/**
 * 节流函数
 * @param {Function} func - 要执行的函数
 * @param {number} limit - 时间限制（毫秒）
 * @returns {Function} - 节流处理后的函数
 */
function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// 导出函数，便于在其他页面使用（如果需要）
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        initMobileMenu,
        initSearch,
        initCategoryFilters,
        initSmoothScroll,
        initAnimations,
        initCardInteractions
    };
}