/* 文档注释 */
/* author: 前端川 */
/* link: www.vsqdc.cn */
/* description: 创意导航网站JavaScript文件，实现交互功能 */

// DOM加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化所有交互功能
    initMobileMenu();
    initSearch();
    initSmoothScroll();
    initCardAnimations();
    initCategoryHover();
});

/**
 * 初始化移动端菜单功能
 * 包括菜单的显示和隐藏
 */
function initMobileMenu() {
    // 获取DOM元素
    const menuToggle = document.getElementById('menuToggle');
    const closeBtn = document.getElementById('closeBtn');
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('overlay');
    const sidebarLinks = document.querySelectorAll('.sidebar-menu a');
    
    // 打开菜单
    menuToggle.addEventListener('click', function() {
        sidebar.classList.add('active');
        overlay.classList.add('active');
        document.body.style.overflow = 'hidden'; // 防止背景滚动
    });
    
    // 关闭菜单
    function closeMenu() {
        sidebar.classList.remove('active');
        overlay.classList.remove('active');
        document.body.style.overflow = 'auto'; // 恢复背景滚动
    }
    
    // 点击关闭按钮关闭菜单
    closeBtn.addEventListener('click', closeMenu);
    
    // 点击遮罩层关闭菜单
    overlay.addEventListener('click', closeMenu);
    
    // 点击菜单链接关闭菜单
    sidebarLinks.forEach(link => {
        link.addEventListener('click', closeMenu);
    });
}

/**
 * 初始化搜索功能
 * 包括搜索框的交互效果
 */
function initSearch() {
    // 获取搜索框元素
    const searchBoxes = document.querySelectorAll('.search-box');
    
    searchBoxes.forEach(searchBox => {
        const input = searchBox.querySelector('input');
        const button = searchBox.querySelector('button');
        
        // 搜索框聚焦效果
        input.addEventListener('focus', function() {
            searchBox.style.boxShadow = '0 4px 12px rgba(108, 92, 231, 0.3)';
        });
        
        // 搜索框失焦效果
        input.addEventListener('blur', function() {
            searchBox.style.boxShadow = '';
        });
        
        // 搜索按钮点击事件
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const searchTerm = input.value.trim();
            if (searchTerm) {
                performSearch(searchTerm);
            }
        });
        
        // 回车键搜索
        input.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                const searchTerm = input.value.trim();
                if (searchTerm) {
                    performSearch(searchTerm);
                }
            }
        });
    });
    
    // 执行搜索
    function performSearch(searchTerm) {
        console.log('搜索:', searchTerm);
        // 这里可以添加实际的搜索逻辑
        alert(`搜索: ${searchTerm}\n\n搜索功能正在开发中...`);
    }
}

/**
 * 初始化平滑滚动
 * 实现锚点链接的平滑滚动效果
 */
function initSmoothScroll() {
    // 获取所有锚点链接
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            // 检查链接是否指向当前页面的锚点
            if (this.getAttribute('href') === '#') return;
            
            // 检查链接是否指向其他页面
            if (this.hostname !== window.location.hostname || this.pathname !== window.location.pathname) {
                return;
            }
            
            e.preventDefault();
            
            // 获取目标元素
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                // 计算滚动位置（考虑导航栏高度）
                const navbarHeight = document.querySelector('.navbar').offsetHeight;
                const targetPosition = targetElement.getBoundingClientRect().top + window.pageYOffset - navbarHeight;
                
                // 平滑滚动
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }
        });
    });
}

/**
 * 初始化卡片动画
 * 当卡片进入视口时添加动画效果
 */
function initCardAnimations() {
    // 获取所有资源卡片
    const resourceCards = document.querySelectorAll('.resource-card, .recommendation-card');
    
    // 创建Intersection Observer实例
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                // 卡片进入视口，添加动画类
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
                observer.unobserve(entry.target); // 只观察一次
            }
        });
    }, {
        threshold: 0.1, // 当卡片10%进入视口时触发
        rootMargin: '0px 0px -50px 0px' // 提前50px触发
    });
    
    // 初始设置卡片样式
    resourceCards.forEach(card => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(30px)';
        card.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(card); // 观察卡片
    });
}

/**
 * 初始化分类项悬停效果
 * 为分类项添加悬停动画
 */
function initCategoryHover() {
    // 获取所有分类项
    const categoryItems = document.querySelectorAll('.category-item');
    
    categoryItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            // 添加脉冲动画
            this.style.animation = 'pulse 0.6s ease-in-out';
            
            // 动画结束后移除动画类，以便下次悬停时再次触发
            setTimeout(() => {
                this.style.animation = '';
            }, 600);
        });
    });
}

/**
 * 滚动监听函数
 * 用于实现导航栏的滚动效果
 */
window.addEventListener('scroll', function() {
    const navbar = document.querySelector('.navbar');
    
    // 当滚动超过50px时，添加滚动样式
    if (window.scrollY > 50) {
        navbar.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.15)';
        navbar.style.padding = '10px 0';
    } else {
        navbar.style.boxShadow = '0 4px 6px rgba(0, 0, 0, 0.1)';
        navbar.style.padding = '15px 0';
    }
});

/**
 * 防抖函数
 * 用于优化高频触发的事件
 * @param {Function} func - 要执行的函数
 * @param {number} delay - 延迟时间（毫秒）
 * @returns {Function} - 防抖处理后的函数
 */
function debounce(func, delay) {
    let timeoutId;
    return function(...args) {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => func.apply(this, args), delay);
    };
}

/**
 * 节流函数
 * 用于限制函数的执行频率
 * @param {Function} func - 要执行的函数
 * @param {number} limit - 时间限制（毫秒）
 * @returns {Function} - 节流处理后的函数
 */
function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

/**
 * 格式化数字
 * @param {number} num - 要格式化的数字
 * @returns {string} - 格式化后的数字字符串
 */
function formatNumber(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
}

/**
 * 获取随机颜色
 * @returns {string} - 随机颜色的十六进制值
 */
function getRandomColor() {
    const letters = '0123456789ABCDEF';
    let color = '#';
    for (let i = 0; i < 6; i++) {
        color += letters[Math.floor(Math.random() * 16)];
    }
    return color;
}

/**
 * 检查设备类型
 * @returns {string} - 设备类型（mobile, tablet, desktop）
 */
function getDeviceType() {
    const width = window.innerWidth;
    if (width < 768) {
        return 'mobile';
    } else if (width < 992) {
        return 'tablet';
    } else {
        return 'desktop';
    }
}

/**
 * 复制文本到剪贴板
 * @param {string} text - 要复制的文本
 * @returns {Promise<boolean>} - 是否复制成功
 */
async function copyToClipboard(text) {
    try {
        await navigator.clipboard.writeText(text);
        return true;
    } catch (err) {
        console.error('无法复制文本:', err);
        return false;
    }
}

/**
 * 显示通知消息
 * @param {string} message - 通知内容
 * @param {string} type - 通知类型（success, error, info）
 * @param {number} duration - 显示时长（毫秒）
 */
function showNotification(message, type = 'info', duration = 3000) {
    // 创建通知元素
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    // 添加样式
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        z-index: 9999;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    `;
    
    // 根据类型设置背景色
    const bgColors = {
        success: '#00b894',
        error: '#d63031',
        info: '#6c5ce7'
    };
    notification.style.backgroundColor = bgColors[type] || bgColors.info;
    
    // 添加到页面
    document.body.appendChild(notification);
    
    // 显示通知
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // 隐藏通知
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateX(100%)';
        
        // 移除元素
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, duration);
}

// 导出公共函数（如果需要）
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        initMobileMenu,
        initSearch,
        initSmoothScroll,
        initCardAnimations,
        initCategoryHover,
        debounce,
        throttle,
        formatNumber,
        getRandomColor,
        getDeviceType,
        copyToClipboard,
        showNotification
    };
}